
/*

Reed1 <-----1 PB0      PA0 20-------> VRefHigh
Reed2 <-----2 PB1   T  PA1 19-------> VRefHigh
Reed3 <-----3 PB2   i  PA2 18-------> DataOut
Reed4 <-----4 PB3   n  PA3 17-------> VRef
            5 VCC   y AGND 16
            6 GND   8 AVCC 15
Reed5 <-----7 PB4   6  PA4 14-------> BattMon
Reed6 <-----8 PB5   1  PA5 13-------> BattJP3
Reed7 <-----9 PB6      PA6 12-------> BattJP2
           10 RESET    PA7 11-------> BattMonEn
                               
fuses: BODLEVEL=111 CKDIV8=1 SUT=10 CKSEL=0011 hfuse=11011111 (0xdf) lfuse=11100011 (0xe3)
*/

#define F_CPU 128000
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>


unsigned char vref_timer = 240, vref_timer2 = 2;


static void delay_ms(unsigned char ms) {
  while( ms-- )
    _delay_ms(1);
}

ISR(WDT_vect) {
}

/*
  BattJP2/BattJP3 set the battery type.
  If BattJP2 is shorted to ground, battery type is 9V alkaline (6-9V).
  If BattJP3 is shorted to ground, battery type is 4 x AA alkaline (4-6V).
  If neither are shorted, battery type is 12V lead acid (12.0-12.9V).
  If both are shorted, battery type is 4 x AA NiMH (3.6-4.8V).
  If BattJP2 is shorted to BattJP3, battery type is 9V NiMH (5.4-7.2V).
*/

#define ADC_FULL_READING 18.0
static unsigned char get_battery_limits(unsigned short* min, unsigned short* max, unsigned short* turnoff) {
  unsigned char state;
  PORTA |= _BV(PA5)|_BV(PA6);
  asm volatile("nop");
  asm volatile("nop");
  state = PINA;
  PORTA &= ~(_BV(PA5)|_BV(PA6));
  *turnoff = (unsigned short)( 3.0 * 1024 / ADC_FULL_READING);
  if( !(state&_BV(PA6)) ) {
    if( !(state&_BV(PA5)) ) {
      // 4 x AA NiMH
      *min = (unsigned short)( 3.6 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 4.8 * 1024 / ADC_FULL_READING);
      *turnoff = (unsigned short)( 3.4 * 1024 / ADC_FULL_READING);
      return 1;
    } else {
      // 9V alkaline
      *min = (unsigned short)( 6.0 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 9.0 * 1024 / ADC_FULL_READING);
      return 2;
    }
  } else {
    if( !(state&_BV(PA5)) ) {
      // 4 x AA alkaline
      *min = (unsigned short)( 4.0 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 6.0 * 1024 / ADC_FULL_READING);
      return 3;
    } else {
      DDRA |= _BV(PA6);
      PORTA |= _BV(PA5);
      state = PINA;
      DDRA &= ~_BV(PA6);
      PORTA &= ~_BV(PA5);

      if( state&_BV(PA5) ) {
        // 12V lead-acid
        *min = (unsigned short)(12.0 * 1024 / ADC_FULL_READING);
        *max = (unsigned short)(12.9 * 1024 / ADC_FULL_READING);
        *turnoff = (unsigned short)(11.5 * 1024 / ADC_FULL_READING);
        return 4;
      } else {
        // 9V NiMH
        *min = (unsigned short)( 5.4 * 1024 / ADC_FULL_READING);
        *max = (unsigned short)( 7.2 * 1024 / ADC_FULL_READING);
        *turnoff = (unsigned short)(5.1 * 1024 / ADC_FULL_READING);
        return 5;
      }
    }
  }
}

// return battery level as a number 0-15 where 0=flat and 15=full
// We expect the BattMonEn (PA7) output to be high before this function is called.
static unsigned char get_battery_level(unsigned char* shutdown) {
  unsigned short min, max, turnoff;
  union {
    unsigned char bytes[2];
    unsigned short word;
  } reading;

  // turn on ADC and start a conversion
  ADCSRA = _BV(ADEN);
  ADCSRA |= _BV(ADSC);
  // work out minimum and maximum expected battery voltages depending upon state of BattJP2 and BattJP3
  get_battery_limits(&min, &max, &turnoff);
  // wait for conversion to finish
  while( ADCSRA&_BV(ADSC) )
    ;
  // get ADC reading
  reading.bytes[0] = ADCL;
  reading.bytes[1] = ADCH;
  // turn off ADC
  ADCSRA = 0;
  if( shutdown )
    *shutdown = (reading.word <= turnoff);
  // convert raw reading to 0-15 battery state based on minimum and maximum expected readings
  if( reading.word < min )
    return 0;
  else if( reading.word > max )
    return 15;
  else
    return ((reading.word - min) * 15 + ((max - min)>>1)) / (max - min);
}

static void send_packet(unsigned char* data, unsigned char num_bytes) {
  while( num_bytes-- ) {
    unsigned char num_bits, packet;
    packet = *data;
    ++data;
    for( num_bits = 8; num_bits; --num_bits ) {
      if( packet&1 ) {
        _delay_ms(0.085);
//        _delay_ms(0.045);
        PORTA |= _BV(PA2);
        _delay_ms(0.215);
//        _delay_ms(0.195);
      } else {
        PORTA &= ~_BV(PA2);
        _delay_ms(0.3);
//        _delay_ms(0.24);
      }
      packet >>= 1;
    }
  }
  PORTA &= ~_BV(PA2);
}

static unsigned char biphase_encode_nybble(unsigned char nybble, unsigned char* plast_state) {
  unsigned char ret = 0;
  unsigned char num_bits;
  for( num_bits = 4; num_bits; --num_bits ) {
    ret >>= 2;
    if( nybble&1 ) {
      if( *plast_state ) {
        ret |= 0x80;
      } else {
        ret |= 0x40;
      }
    } else {
      if( !*plast_state ) {
        ret |= 0xC0;
      }
      *plast_state ^= 1;
    }
    nybble >>= 1;
  }
  return ret;
}

/*
  Packet format:

              1111111111222222222233
    01234567890123456789012345678901
    11aaaabbbbcccccccccccccccddddddd
    ....____....____....____....____
    
    a = rudder position (4 bits)
    b = battery level (4 bits)
    c = unique ID (15 bits)
    d = checksum (7 bits)
*/

static unsigned char CRC7(unsigned char* data, unsigned char num_bytes) {
  unsigned char crc, datum, i;
  crc = 0;
  while( num_bytes ) {
    datum = *data;
    for( i=0; i<8; ++i ) {
      crc <<= 1;
      if( (datum&0x80)^(crc&0x80) )
        crc ^= 9;
      datum <<= 1;
    }
    ++data;
    --num_bytes;
  }
  return crc & 0x7f;
}

static void form_packet(unsigned char* packet, unsigned char rudder_pos, unsigned char batt_level, unsigned short unique_id) {
  unsigned char crc;
  unsigned char last_state = 0;

  rudder_pos &= 0x0f;
  batt_level &= 0x0f;
  unique_id &= 0x7fff;

  packet[0] = rudder_pos|(batt_level<<4);
  packet[1] = unique_id&255;
  packet[2] = unique_id>>8;
  crc = CRC7(packet, 3);
  packet[0] = biphase_encode_nybble( 3|(rudder_pos<<2), &last_state );
  packet[1] = biphase_encode_nybble( (rudder_pos>>2)|(batt_level<<2), &last_state );
  packet[2] = biphase_encode_nybble( (batt_level>>2)|(unique_id<<2), &last_state );
  packet[3] = biphase_encode_nybble( (unique_id>>2), &last_state );
  packet[4] = biphase_encode_nybble( (unique_id>>6), &last_state );
  packet[5] = biphase_encode_nybble( (unique_id>>10), &last_state );
  packet[6] = biphase_encode_nybble( (unique_id>>14)|(crc<<1), &last_state );
  packet[7] = biphase_encode_nybble( (crc>>3), &last_state );
}

static unsigned short generate_unique_id() {
  unsigned char i, j;
  unsigned short ret;
  union {
    unsigned char bytes[2];
    unsigned short word;
  } reading;

  ADMUX = _BV(REFS0)|8;

  // turn on ADC
  ADCSRA = _BV(ADEN);
  ret = 0;
  for( i = 0; i < 15; ++i ) {
    // start a conversion
    ADCSRA |= _BV(ADSC);
    // wait for conversion to finish
    while( ADCSRA&_BV(ADSC) )
      ;
    // get ADC reading
    reading.bytes[0] = ADCL;
    reading.bytes[1] = ADCH;
    ret <<= 1;
    for( j = 0; j < 16; ++j ) {
      ret ^= reading.word&1;
      reading.word >>= 1;
    }
  }
  // turn off ADC
  ADCSRA = 0;

  ADMUX = _BV(REFS0)|_BV(MUX1)|_BV(MUX0);

  return ret;
}

int main(void) {
  unsigned char last_switch_state = 0xFF, switch_state, battlevel, shutdown;
  unsigned short unique_id;

  // enable power down sleep mode
  MCUCR = _BV(SE)|_BV(SM1);
  // turn on voltage reference (to allow user to calibrate) and battery sense enable output
  DDRA = _BV(PA0)|_BV(PA1)|_BV(PA2)|_BV(PA7);
  PORTA = _BV(PA0)|_BV(PA1);
  // set up watchdog timer for 1/8 second intervals
  WDTCR = _BV(WDP1)|_BV(WDP0);
  // set up ADC: select PA4 (ADC3) as ADC input and PA3 (AREF) as voltage reference
  ADMUX = _BV(REFS0)|_BV(MUX1)|_BV(MUX0);
  // disable digital input for AREF and ADC3
  DIDR0 = _BV(AREFD)|_BV(ADC3D);
  // turn on ADC initially, to allow for accurate Vref calibration
  ADCSRA = _BV(ADEN);
/*
  delay_ms(250);
  delay_ms(250);
  delay_ms(250);
  delay_ms(250);
  PORTA |= _BV(PA7);
  delay_ms(100);
  battlevel = get_battery_level(&shutdown);
  PORTA &= ~_BV(PA7);
  while( battlevel ) {
    delay_ms(250);
    delay_ms(250);
    PORTA |= _BV(PA7);
    delay_ms(100);
    PORTA &= ~_BV(PA7);
    --battlevel;
  }
*/
  unique_id = generate_unique_id();

  while(1) {
    PORTB = 0x7F;
    asm volatile("nop");
    switch_state = PINB;
    PORTB = 0;
    switch_state &= 0x7F;
    if( last_switch_state != switch_state ) {
      unsigned char switches_closed = last_switch_state & ~switch_state;
      unsigned char special_case = (!(last_switch_state&8) && (switch_state&8));
      last_switch_state = switch_state;
      if( switches_closed || special_case ) {
        unsigned char rudder_pos;
        for( rudder_pos = 0; rudder_pos < 7; ++rudder_pos ) {
          if( switches_closed&1 )
            break;
          switches_closed >>= 1;
        }
        if( !(switches_closed&~1) || special_case ) {
          unsigned char batt_level, transmissions;
          PORTA |= _BV(PA7);
          DDRA |= _BV(PA0)|_BV(PA1);
          PORTA |= _BV(PA0)|_BV(PA1);
          if( switches_closed&~1 ) {
            rudder_pos = 7;
            last_switch_state |= 0x08;
          }
          batt_level = get_battery_level(&shutdown);
          if( !shutdown ) {
            unsigned char packet[8];
            form_packet(packet, rudder_pos, batt_level, unique_id);
            // warm up the transmitter
            _delay_ms(0.8);
            PORTA |= _BV(PA2);
            _delay_ms(0.8);
            PORTA &= ~_BV(PA2);
            _delay_ms(0.8);
            transmissions = 5;//10;
            while( transmissions-- ) {
              send_packet(packet, sizeof(packet));
              if( transmissions )
                _delay_ms(2);
            }
          }
          PORTA &= ~_BV(PA7);
          if( !vref_timer2 ) {
            DDRA &= ~(_BV(PA0)|_BV(PA1));
            PORTA &= ~(_BV(PA0)|_BV(PA1));
          }
        }
      }
    }
    if( vref_timer ) {
      if( --vref_timer == 0 ) {
        if( --vref_timer2 == 0 ) {
          DDRA &= ~(_BV(PA0)|_BV(PA1));
          PORTA &= ~(_BV(PA0)|_BV(PA1));
          // turn off ADC as well as reference supply, to save power
          ADCSRA = 0;
        } else {
          vref_timer = 240;
        }
      }
    }

    WDTCR |= _BV(WDIE)|_BV(WDIF);
    sei();
    asm volatile("sleep");
  }
  return 0;
}
